// Created with Squiffy 5.1.0
// https://github.com/textadventures/squiffy

(function(){
/* jshint quotmark: single */
/* jshint evil: true */

var squiffy = {};

(function () {
    'use strict';

    squiffy.story = {};

    var initLinkHandler = function () {
        var handleLink = function (link) {
            if (link.hasClass('disabled')) return;
            var passage = link.data('passage');
            var section = link.data('section');
            var rotateAttr = link.attr('data-rotate');
            var sequenceAttr = link.attr('data-sequence');
            if (passage) {
                disableLink(link);
                squiffy.set('_turncount', squiffy.get('_turncount') + 1);
                passage = processLink(passage);
                if (passage) {
                    currentSection.append('<hr/>');
                    squiffy.story.passage(passage);
                }
                var turnPassage = '@' + squiffy.get('_turncount');
                if (turnPassage in squiffy.story.section.passages) {
                    squiffy.story.passage(turnPassage);
                }
                if ('@last' in squiffy.story.section.passages && squiffy.get('_turncount')>= squiffy.story.section.passageCount) {
                    squiffy.story.passage('@last');
                }
            }
            else if (section) {
                currentSection.append('<hr/>');
                disableLink(link);
                section = processLink(section);
                squiffy.story.go(section);
            }
            else if (rotateAttr || sequenceAttr) {
                var result = rotate(rotateAttr || sequenceAttr, rotateAttr ? link.text() : '');
                link.html(result[0].replace(/&quot;/g, '"').replace(/&#39;/g, '\''));
                var dataAttribute = rotateAttr ? 'data-rotate' : 'data-sequence';
                link.attr(dataAttribute, result[1]);
                if (!result[1]) {
                    disableLink(link);
                }
                if (link.attr('data-attribute')) {
                    squiffy.set(link.attr('data-attribute'), result[0]);
                }
                squiffy.story.save();
            }
        };

        squiffy.ui.output.on('click', 'a.squiffy-link', function () {
            handleLink(jQuery(this));
        });

        squiffy.ui.output.on('keypress', 'a.squiffy-link', function (e) {
            if (e.which !== 13) return;
            handleLink(jQuery(this));
        });

        squiffy.ui.output.on('mousedown', 'a.squiffy-link', function (event) {
            event.preventDefault();
        });
    };

    var disableLink = function (link) {
        link.addClass('disabled');
        link.attr('tabindex', -1);
    }
    
    squiffy.story.begin = function () {
        if (!squiffy.story.load()) {
            squiffy.story.go(squiffy.story.start);
        }
    };

    var processLink = function(link) {
		link = String(link);
        var sections = link.split(',');
        var first = true;
        var target = null;
        sections.forEach(function (section) {
            section = section.trim();
            if (startsWith(section, '@replace ')) {
                replaceLabel(section.substring(9));
            }
            else {
                if (first) {
                    target = section;
                }
                else {
                    setAttribute(section);
                }
            }
            first = false;
        });
        return target;
    };

    var setAttribute = function(expr) {
        var lhs, rhs, op, value;
        var setRegex = /^([\w]*)\s*=\s*(.*)$/;
        var setMatch = setRegex.exec(expr);
        if (setMatch) {
            lhs = setMatch[1];
            rhs = setMatch[2];
            if (isNaN(rhs)) {
				if(startsWith(rhs,"@")) rhs=squiffy.get(rhs.substring(1));
                squiffy.set(lhs, rhs);
            }
            else {
                squiffy.set(lhs, parseFloat(rhs));
            }
        }
        else {
			var incDecRegex = /^([\w]*)\s*([\+\-\*\/])=\s*(.*)$/;
            var incDecMatch = incDecRegex.exec(expr);
            if (incDecMatch) {
                lhs = incDecMatch[1];
                op = incDecMatch[2];
				rhs = incDecMatch[3];
				if(startsWith(rhs,"@")) rhs=squiffy.get(rhs.substring(1));
				rhs = parseFloat(rhs);
                value = squiffy.get(lhs);
                if (value === null) value = 0;
                if (op == '+') {
                    value += rhs;
                }
                if (op == '-') {
                    value -= rhs;
                }
				if (op == '*') {
					value *= rhs;
				}
				if (op == '/') {
					value /= rhs;
				}
                squiffy.set(lhs, value);
            }
            else {
                value = true;
                if (startsWith(expr, 'not ')) {
                    expr = expr.substring(4);
                    value = false;
                }
                squiffy.set(expr, value);
            }
        }
    };

    var replaceLabel = function(expr) {
        var regex = /^([\w]*)\s*=\s*(.*)$/;
        var match = regex.exec(expr);
        if (!match) return;
        var label = match[1];
        var text = match[2];
        if (text in squiffy.story.section.passages) {
            text = squiffy.story.section.passages[text].text;
        }
        else if (text in squiffy.story.sections) {
            text = squiffy.story.sections[text].text;
        }
        var stripParags = /^<p>(.*)<\/p>$/;
        var stripParagsMatch = stripParags.exec(text);
        if (stripParagsMatch) {
            text = stripParagsMatch[1];
        }
        var $labels = squiffy.ui.output.find('.squiffy-label-' + label);
        $labels.fadeOut(1000, function() {
            $labels.html(squiffy.ui.processText(text));
            $labels.fadeIn(1000, function() {
                squiffy.story.save();
            });
        });
    };

    squiffy.story.go = function(section) {
        squiffy.set('_transition', null);
        newSection();
        squiffy.story.section = squiffy.story.sections[section];
        if (!squiffy.story.section) return;
        squiffy.set('_section', section);
        setSeen(section);
        var master = squiffy.story.sections[''];
        if (master) {
            squiffy.story.run(master);
            squiffy.ui.write(master.text);
        }
        squiffy.story.run(squiffy.story.section);
        // The JS might have changed which section we're in
        if (squiffy.get('_section') == section) {
            squiffy.set('_turncount', 0);
            squiffy.ui.write(squiffy.story.section.text);
            squiffy.story.save();
        }
    };

    squiffy.story.run = function(section) {
        if (section.clear) {
            squiffy.ui.clearScreen();
        }
        if (section.attributes) {
            processAttributes(section.attributes);
        }
        if (section.js) {
            section.js();
        }
    };

    squiffy.story.passage = function(passageName) {
        var passage = squiffy.story.section.passages[passageName];
        if (!passage) return;
        setSeen(passageName);
        var masterSection = squiffy.story.sections[''];
        if (masterSection) {
            var masterPassage = masterSection.passages[''];
            if (masterPassage) {
                squiffy.story.run(masterPassage);
                squiffy.ui.write(masterPassage.text);
            }
        }
        var master = squiffy.story.section.passages[''];
        if (master) {
            squiffy.story.run(master);
            squiffy.ui.write(master.text);
        }
        squiffy.story.run(passage);
        squiffy.ui.write(passage.text);
        squiffy.story.save();
    };

    var processAttributes = function(attributes) {
        attributes.forEach(function (attribute) {
            if (startsWith(attribute, '@replace ')) {
                replaceLabel(attribute.substring(9));
            }
            else {
                setAttribute(attribute);
            }
        });
    };

    squiffy.story.restart = function() {
        if (squiffy.ui.settings.persist && window.localStorage) {
            var keys = Object.keys(localStorage);
            jQuery.each(keys, function (idx, key) {
                if (startsWith(key, squiffy.story.id)) {
                    localStorage.removeItem(key);
                }
            });
        }
        else {
            squiffy.storageFallback = {};
        }
        if (squiffy.ui.settings.scroll === 'element') {
            squiffy.ui.output.html('');
            squiffy.story.begin();
        }
        else {
            location.reload();
        }
    };

    squiffy.story.save = function() {
        squiffy.set('_output', squiffy.ui.output.html());
    };

    squiffy.story.load = function() {
        var output = squiffy.get('_output');
        if (!output) return false;
        squiffy.ui.output.html(output);
        currentSection = jQuery('#' + squiffy.get('_output-section'));
        squiffy.story.section = squiffy.story.sections[squiffy.get('_section')];
        var transition = squiffy.get('_transition');
        if (transition) {
            eval('(' + transition + ')()');
        }
        return true;
    };

    var setSeen = function(sectionName) {
        var seenSections = squiffy.get('_seen_sections');
        if (!seenSections) seenSections = [];
        if (seenSections.indexOf(sectionName) == -1) {
            seenSections.push(sectionName);
            squiffy.set('_seen_sections', seenSections);
        }
    };

    squiffy.story.seen = function(sectionName) {
        var seenSections = squiffy.get('_seen_sections');
        if (!seenSections) return false;
        return (seenSections.indexOf(sectionName) > -1);
    };
    
    squiffy.ui = {};

    var currentSection = null;
    var screenIsClear = true;
    var scrollPosition = 0;

    var newSection = function() {
        if (currentSection) {
            disableLink(jQuery('.squiffy-link', currentSection));
        }
        var sectionCount = squiffy.get('_section-count') + 1;
        squiffy.set('_section-count', sectionCount);
        var id = 'squiffy-section-' + sectionCount;
        currentSection = jQuery('<div/>', {
            id: id,
        }).appendTo(squiffy.ui.output);
        squiffy.set('_output-section', id);
    };

    squiffy.ui.write = function(text) {
        screenIsClear = false;
        scrollPosition = squiffy.ui.output.height();
        currentSection.append(jQuery('<div/>').html(squiffy.ui.processText(text)));
        squiffy.ui.scrollToEnd();
    };

    squiffy.ui.clearScreen = function() {
        squiffy.ui.output.html('');
        screenIsClear = true;
        newSection();
    };

    squiffy.ui.scrollToEnd = function() {
        var scrollTo, currentScrollTop, distance, duration;
        if (squiffy.ui.settings.scroll === 'element') {
            scrollTo = squiffy.ui.output[0].scrollHeight - squiffy.ui.output.height();
            currentScrollTop = squiffy.ui.output.scrollTop();
            if (scrollTo > currentScrollTop) {
                distance = scrollTo - currentScrollTop;
                duration = distance / 0.4;
                squiffy.ui.output.stop().animate({ scrollTop: scrollTo }, duration);
            }
        }
        else {
            scrollTo = scrollPosition;
            currentScrollTop = Math.max(jQuery('body').scrollTop(), jQuery('html').scrollTop());
            if (scrollTo > currentScrollTop) {
                var maxScrollTop = jQuery(document).height() - jQuery(window).height();
                if (scrollTo > maxScrollTop) scrollTo = maxScrollTop;
                distance = scrollTo - currentScrollTop;
                duration = distance / 0.5;
                jQuery('body,html').stop().animate({ scrollTop: scrollTo }, duration);
            }
        }
    };

    squiffy.ui.processText = function(text) {
        function process(text, data) {
            var containsUnprocessedSection = false;
            var open = text.indexOf('{');
            var close;
            
            if (open > -1) {
                var nestCount = 1;
                var searchStart = open + 1;
                var finished = false;
             
                while (!finished) {
                    var nextOpen = text.indexOf('{', searchStart);
                    var nextClose = text.indexOf('}', searchStart);
         
                    if (nextClose > -1) {
                        if (nextOpen > -1 && nextOpen < nextClose) {
                            nestCount++;
                            searchStart = nextOpen + 1;
                        }
                        else {
                            nestCount--;
                            searchStart = nextClose + 1;
                            if (nestCount === 0) {
                                close = nextClose;
                                containsUnprocessedSection = true;
                                finished = true;
                            }
                        }
                    }
                    else {
                        finished = true;
                    }
                }
            }
            
            if (containsUnprocessedSection) {
                var section = text.substring(open + 1, close);
                var value = processTextCommand(section, data);
                text = text.substring(0, open) + value + process(text.substring(close + 1), data);
            }
            
            return (text);
        }

        function processTextCommand(text, data) {
            if (startsWith(text, 'if ')) {
                return processTextCommand_If(text, data);
            }
            else if (startsWith(text, 'else:')) {
                return processTextCommand_Else(text, data);
            }
            else if (startsWith(text, 'label:')) {
                return processTextCommand_Label(text, data);
            }
            else if (/^rotate[: ]/.test(text)) {
                return processTextCommand_Rotate('rotate', text, data);
            }
            else if (/^sequence[: ]/.test(text)) {
                return processTextCommand_Rotate('sequence', text, data);   
            }
            else if (text in squiffy.story.section.passages) {
                return process(squiffy.story.section.passages[text].text, data);
            }
            else if (text in squiffy.story.sections) {
                return process(squiffy.story.sections[text].text, data);
            }
			else if (startsWith(text,'@') && !startsWith(text,'@replace')) {
				processAttributes(text.substring(1).split(","));
				return "";
			}
            return squiffy.get(text);
        }

        function processTextCommand_If(section, data) {
            var command = section.substring(3);
            var colon = command.indexOf(':');
            if (colon == -1) {
                return ('{if ' + command + '}');
            }

            var text = command.substring(colon + 1);
            var condition = command.substring(0, colon);
			condition = condition.replace("<", "&lt;");
            var operatorRegex = /([\w ]*)(=|&lt;=|&gt;=|&lt;&gt;|&lt;|&gt;)(.*)/;
            var match = operatorRegex.exec(condition);

            var result = false;

            if (match) {
                var lhs = squiffy.get(match[1]);
                var op = match[2];
                var rhs = match[3];

				if(startsWith(rhs,'@')) rhs=squiffy.get(rhs.substring(1));
				
                if (op == '=' && lhs == rhs) result = true;
                if (op == '&lt;&gt;' && lhs != rhs) result = true;
                if (op == '&gt;' && lhs > rhs) result = true;
                if (op == '&lt;' && lhs < rhs) result = true;
                if (op == '&gt;=' && lhs >= rhs) result = true;
                if (op == '&lt;=' && lhs <= rhs) result = true;
            }
            else {
                var checkValue = true;
                if (startsWith(condition, 'not ')) {
                    condition = condition.substring(4);
                    checkValue = false;
                }

                if (startsWith(condition, 'seen ')) {
                    result = (squiffy.story.seen(condition.substring(5)) == checkValue);
                }
                else {
                    var value = squiffy.get(condition);
                    if (value === null) value = false;
                    result = (value == checkValue);
                }
            }

            var textResult = result ? process(text, data) : '';

            data.lastIf = result;
            return textResult;
        }

        function processTextCommand_Else(section, data) {
            if (!('lastIf' in data) || data.lastIf) return '';
            var text = section.substring(5);
            return process(text, data);
        }

        function processTextCommand_Label(section, data) {
            var command = section.substring(6);
            var eq = command.indexOf('=');
            if (eq == -1) {
                return ('{label:' + command + '}');
            }

            var text = command.substring(eq + 1);
            var label = command.substring(0, eq);

            return '<span class="squiffy-label-' + label + '">' + process(text, data) + '</span>';
        }

        function processTextCommand_Rotate(type, section, data) {
            var options;
            var attribute = '';
            if (section.substring(type.length, type.length + 1) == ' ') {
                var colon = section.indexOf(':');
                if (colon == -1) {
                    return '{' + section + '}';
                }
                options = section.substring(colon + 1);
                attribute = section.substring(type.length + 1, colon);
            }
            else {
                options = section.substring(type.length + 1);
            }
            var rotation = rotate(options.replace(/"/g, '&quot;').replace(/'/g, '&#39;'));
            if (attribute) {
                squiffy.set(attribute, rotation[0]);
            }
            return '<a class="squiffy-link" data-' + type + '="' + rotation[1] + '" data-attribute="' + attribute + '" role="link">' + rotation[0] + '</a>';
        }

        var data = {
            fulltext: text
        };
        return process(text, data);
    };

    squiffy.ui.transition = function(f) {
        squiffy.set('_transition', f.toString());
        f();
    };

    squiffy.storageFallback = {};

    squiffy.set = function(attribute, value) {
        if (typeof value === 'undefined') value = true;
        if (squiffy.ui.settings.persist && window.localStorage) {
            localStorage[squiffy.story.id + '-' + attribute] = JSON.stringify(value);
        }
        else {
            squiffy.storageFallback[attribute] = JSON.stringify(value);
        }
        squiffy.ui.settings.onSet(attribute, value);
    };

    squiffy.get = function(attribute) {
        var result;
        if (squiffy.ui.settings.persist && window.localStorage) {
            result = localStorage[squiffy.story.id + '-' + attribute];
        }
        else {
            result = squiffy.storageFallback[attribute];
        }
        if (!result) return null;
        return JSON.parse(result);
    };

    var startsWith = function(string, prefix) {
        return string.substring(0, prefix.length) === prefix;
    };

    var rotate = function(options, current) {
        var colon = options.indexOf(':');
        if (colon == -1) {
            return [options, current];
        }
        var next = options.substring(0, colon);
        var remaining = options.substring(colon + 1);
        if (current) remaining += ':' + current;
        return [next, remaining];
    };

    var methods = {
        init: function (options) {
            var settings = jQuery.extend({
                scroll: 'body',
                persist: true,
                restartPrompt: true,
                onSet: function (attribute, value) {}
            }, options);

            squiffy.ui.output = this;
            squiffy.ui.restart = jQuery(settings.restart);
            squiffy.ui.settings = settings;

            if (settings.scroll === 'element') {
                squiffy.ui.output.css('overflow-y', 'auto');
            }

            initLinkHandler();
            squiffy.story.begin();
            
            return this;
        },
        get: function (attribute) {
            return squiffy.get(attribute);
        },
        set: function (attribute, value) {
            squiffy.set(attribute, value);
        },
        restart: function () {
            if (!squiffy.ui.settings.restartPrompt || confirm('Are you sure you want to restart?')) {
                squiffy.story.restart();
            }
        }
    };

    jQuery.fn.squiffy = function (methodOrOptions) {
        if (methods[methodOrOptions]) {
            return methods[methodOrOptions]
                .apply(this, Array.prototype.slice.call(arguments, 1));
        }
        else if (typeof methodOrOptions === 'object' || ! methodOrOptions) {
            return methods.init.apply(this, arguments);
        } else {
            jQuery.error('Method ' +  methodOrOptions + ' does not exist');
        }
    };
})();

var get = squiffy.get;
var set = squiffy.set;


squiffy.story.start = '_default';
squiffy.story.id = '6d2e819500';
squiffy.story.sections = {
	'_default': {
		'text': "<p>Once there was a very old witch who lived by herself in a little shack at the top of a hill. In her youth she had been quite a sight to see, smiting and blighting and turning peoples children into rodents. But now she was older and there had been four live-action Chipmunk movies and that was a cruel enough thing to inflict on children. The very old witch was very old; she had seen humans do a lot of cruel things to each other and now she didn&#39;t know what else she could do to the world anymore. So the very old witch passed her time quietly, rocking in her wicker chair on the front porch, glaring at <a class=\"squiffy-link link-section\" data-section=\"passersby\" role=\"link\" tabindex=\"0\">passersby</a>.</p>",
		'passages': {
		},
	},
	'passersby': {
		'text': "<p>The people of the village did not usually bother the very old witch. It had been a very long time since she had cursed any of them. Presently the only folk who still feared her were the tax collectors. This was because the last collector to visit the very old witch had never left. He was now made of a weather resistant plastic and stood in the very old witch&#39;s yard, with pretty blue and purple morning glories twining up his legs. He had told the very old witch his name was Donald, but she hated that name. Nowadays she called him Hernando.</p>\n<p>The very old witch did not do much these days. She <a class=\"squiffy-link link-passage\" data-passage=\"tended her garden\" role=\"link\" tabindex=\"0\">tended her garden</a>... most of the time. She used her magic to <a class=\"squiffy-link link-passage\" data-passage=\"steal newspapers and magazines\" role=\"link\" tabindex=\"0\">steal newspapers and magazines</a> at a healthy distance from town. She made <a class=\"squiffy-link link-passage\" data-passage=\"grilled cheese\" role=\"link\" tabindex=\"0\">grilled cheese</a> for meals, when she could be bothered to eat. </p>",
		'passageCount': 3,
		'passages': {
			'tended her garden': {
				'text': "<p>The very old witch had once grown belladonna and nightshade and those days were mostly long gone. She did keep a single tomato plant, gnarled and yellowing, for old times&#39; sake. She also grew a number of root vegetables, because they were slightly less work than flimsy cornstalks or patchy melons or berries that had to be picked every week.</p>",
			},
			'steal newspapers and magazines': {
				'text': "<p>If you were to peek in the very old witch&#39;s windows you would see stacks of fishing magazines, science digests, newsletters from the Commission on Aging and the local elementary school and the Catholic church. Bales of the latter, actually--the truck carrying the weekly bulletins had crashed into a fire truck under the town&#39;s single stoplight, which had amused the very old witch so much that she magicked away the bales of bulletins as a keepsake of that wild, wild day.</p>",
			},
			'grilled cheese': {
				'text': "<p>The very old witch made her sandwiches from bread she&#39;d magicked away from the grocery store&#39;s truck and cheese from the dollar store. The cheese tasted very much like plastic, or possibly like bricks of pure cancer cells.</p>",
			},
			'@last': {
				'text': "<p>It was... a <a class=\"squiffy-link link-section\" data-section=\"quiet life\" role=\"link\" tabindex=\"0\">quiet life</a>.</p>",
			},
		},
	},
	'quiet life': {
		'text': "<p>The very old witch awoke one day particularly sand and grumpy. She didn&#39;t want grilled cheese. She didn&#39;t want to read her newspapers. She went out to the front yard and glared down the hill at the town. Although her eyes weren&#39;t as good as they had once been, she could see people going about their business. It irritated her.</p>\n<p>The very old witch felt restless, as if something were missing. But what could it be?</p>\n<p>Perhaps her long life was coming to a close, she thought, leaning up against Hernando to give her corns a rest, and this was her one last hurrah before she went.</p>\n<p>Perhaps, she thought, she wanted to <a class=\"squiffy-link link-section\" data-section=\"blight\" role=\"link\" tabindex=\"0\">blight</a> something.</p>",
		'passages': {
		},
	},
	'blight': {
		'text': "<p>So the very old witch cast her eyes around and found a field of prize winning dahlias, as big as dinner plates, winking merrily in the wun. With a flick of her wrist she brought down swarms of Japanese beetles to feast on the flowers. The field turned brown and dead beneath them in a matter of moments.</p>\n<p>But still the very old witch felt restless.</p>\n<p>Perhaps <a class=\"squiffy-link link-section\" data-section=\"smiting\" role=\"link\" tabindex=\"0\">smiting</a> will help, she thought.</p>",
		'passages': {
		},
	},
	'smiting': {
		'text': "<p>As it happened a land developer was driving past in his shiny red sports car. He was whistling to himself and thinking about the money he would make on his next job, when he tore down low-income housing to build designer cupcake stores.</p>\n<p>With a click of her fingers the very old witch opened up a pothole in front of him. The shiny red sports car crashed into it, and the land developer, who was not wearing a seat belt, flew through the windshield and rolled down the hill.</p>\n<p>But the very old witch was still <a class=\"squiffy-link link-section\" data-section=\"restless\" role=\"link\" tabindex=\"0\">restless</a>.</p>",
		'passages': {
		},
	},
	'restless': {
		'text': "<p>The very old witch turned her eyes upon the town, seeking some one knew to smite. Down below a tall person and a small person were walking up to the post office. They would have to do.</p>\n<p>The very old witch rolled up her sleeves and got ready to cast a lightning spell or tornado spell and just as she was about to shout the magic words the small person turned and she saw that it was a child.</p>\n<p>She lowered her arms and watched as the child rushed back and crouched down beside the sidewalk. After a moment&#39;s fidgeting, the child held up a leaf. The adult patted them on the head and lead them indoors. The very old witch did not smite them.</p>\n<p><a class=\"squiffy-link link-section\" data-section=\"A child\" role=\"link\" tabindex=\"0\">A child</a>, she thought. Surely that&#39;s what is missing.</p>",
		'passages': {
		},
	},
	'A child': {
		'text': "<p>The very old witch hobbled quickly to her kitchen and found {sequence:a satchet of lavender:a vial of crushed ogre&#39;s bones:<a class=\"squiffy-link link-section\" data-section=\"a handful of magic dust and a hickory wand\" role=\"link\" tabindex=\"0\">a handful of magic dust and a hickory wand</a>}</p>",
		'passages': {
		},
	},
	'a handful of magic dust and a hickory wand': {
		'text': "<p>which she carried off to her garden. </p>\n<p>There she scuffed some dirt into approximately the shape of a torso. It was lumpy on one side and the legs were sort of crooked, but the very old witch had never claimed to be good at the human form. She said a few magic words over the torso and dug around the garden until she found something resembling a head. It was a turnip, a large-ish turnip that was still a little too small for the torso. But the very old witch had never claimed to be good at proportions, so she sprinkled her lavender and bones over the turnip and stuck it on the torso and then waved the wand and the dust over everything.</p>\n<p>And then... <a class=\"squiffy-link link-section\" data-section=\"she waited\" role=\"link\" tabindex=\"0\">she waited</a>.</p>",
		'passages': {
		},
	},
	'she waited': {
		'text': "<p>She waited and waited. She waited as the sun went down and the moon came up and a thin irritating layer of dew formed over everything. She waited until at last the turnip and the torso stirred, and then sat up. The turnip-headed golum looked all around her. She took in the stars in the sky and the chirping of the chicadas and the fiery wreck of the land developer&#39;s sports car, which was still flaming in the pothole. Finally she turned and saw the very old witch. A wrinkle that looked very much like a smile formed on her turnip face.</p>\n<p>&quot;Hello,&quot; said the turnip girl.</p>\n<p>It was with a sinking feeling that the very old witch realized she didn&#39;t feel any less empty inside. This hadn&#39;t been <a class=\"squiffy-link link-section\" data-section=\"what was missing\" role=\"link\" tabindex=\"0\">what was missing</a> after all.</p>",
		'passages': {
		},
	},
	'what was missing': {
		'clear': true,
		'text': "<p>The very old witch reluctantly opened her home unto the turnip girl, not because she wanted to, but because she had created this life and it was her responsibility to take care of her.</p>\n<p>The turnip girl was always asking questions. &quot;Why does the sun shine?&quot; she wanted to know. &quot;Why is the grass green? Why don&#39;t we ever go down to the village?&quot;</p>\n<p>The very old witch tried to <i><a class=\"squiffy-link link-section\" data-section=\"answer her questions, listen=yes\" role=\"link\" tabindex=\"0\">answer her questions</a></i>, but sometimes <i><a class=\"squiffy-link link-section\" data-section=\"didn't feel like it, listen=no\" role=\"link\" tabindex=\"0\">she just didn&#39;t feel like it</a></i>.</p>",
		'passages': {
		},
	},
	'answer her questions': {
		'text': "<p>&quot;People are terrible,&quot; she told the turnip girl. &quot;They are always hurting one another or themselves. I used to smite and blight them, but they could always do it much more effeciently than I could. Do you understand?&quot;</p>\n<p>&quot;No,&quot; said the turnip girl. &quot;Why can&#39;t we go visit them?&quot;</p>\n<p>&quot;When you&#39;re older you can visit all you like,&quot; the very old witch said. &quot;Until then you can just stay here with me and Hernando.&quot;</p>\n<p>The turnip girl pouted for a bit, but Hernando was very fun to climb on, and the village was awfully far to go <a class=\"squiffy-link link-section\" data-section=\"alone\" role=\"link\" tabindex=\"0\">alone</a>.</p>",
		'passages': {
		},
	},
	'didn\'t feel like it': {
		'text': "<p>&quot;We just don&#39;t, that&#39;s all,&quot; the very old witch said, flipping absently through a mail order catalogue. Unfortunately the catalogue seemed to be selling rubber gardening boots and inexpensive toilet plungers and other boring things. She tossed the catalogue back on the stack. &quot;Let them do their thing and we will do ours. That&#39;s all I want to hear on the matter.&quot;</p>\n<p>The turnip girl didn&#39;t ask <a class=\"squiffy-link link-section\" data-section=\"alone\" role=\"link\" tabindex=\"0\">again</a>.</p>",
		'passages': {
		},
	},
	'alone': {
		'text': "<p>When the turnip girl was a little older she asked the very old witch for books, and the very old witch obliged by magicking books out of the library&#39;s drop box. It amused her to picture the library patrons gnashing their teeth, pleading with the librarians that they had turned in their books, honest, someone must have taken them out of the box and that was why the books were late. It amused her so much that at first she didn&#39;t give too much thought to what the turnip girl was reading.</p>\n<p>But if she didn&#39;t know at first she soon found out. The turnip girl was full of even more questions. This time the questions were more difficult to answer.</p>\n<p>&quot;Why do people make friends?&quot; she asked. &quot;Why do they fall in love? Why do they get so attached to pets?&quot; she asked, waving around an award-winning book about a boy who shot his dog. Every time the very old witch looked it seemed to be a different book with the same plot.</p>\n<p>The truth was, the very old witch <i><a class=\"squiffy-link link-section\" data-section=\"dunno, explain=cant\" role=\"link\" tabindex=\"0\">didn&#39;t know</a></i>, nor <i><a class=\"squiffy-link link-section\" data-section=\"don't care, explain=wont\" role=\"link\" tabindex=\"0\">did she care</a></i></p>",
		'passages': {
		},
	},
	'dunno': {
		'text': "<p>&quot;I don&#39;t know,&quot; the very old witch said. &quot;It must make them happy.&quot;</p>\n<p>&quot;But then why are they mean to each other?&quot; the turnip girl took out another book, a book about wars and bombs and civilian casualties. &quot;That shouldn&#39;t make them happy. That should make them sad.&quot;</p>\n<p>And the very old witch could only <a class=\"squiffy-link link-section\" data-section=\"shrug\" role=\"link\" tabindex=\"0\">shrug</a>.</p>",
		'passages': {
		},
	},
	'don\'t care': {
		'text': "<p>&quot;What difference does it make?&quot; the very old witch asked. &quot;You aren&#39;t people.&quot;</p>\n<p>The turnip girl looked a little sad at this, at least as much as a turnip girl can, and {if listen=no:again} she <a class=\"squiffy-link link-section\" data-section=\"shrug\" role=\"link\" tabindex=\"0\">stopped asking</a>.</p>",
		'passages': {
		},
	},
	'shrug': {
		'text': "<p>One day, several years after the very old witch first got the feeling that something was missing, the turnip girl came to her with a plan.</p>\n<p>&quot;I&#39;m not a little turnip girl anymore,&quot; she said. &quot;I&#39;m a turnip young adult, and I think it&#39;s time for me to make my own way. So I&#39;m going to enroll in college.&quot;</p>\n<p>This was good, the very old witch told herself. It wasn&#39;t as if she wanted to hang around, taking care of this vegetable child who didn&#39;t quite fill up the empty feeling. But what should she say? <i><a class=\"squiffy-link link-section\" data-section=\"safe, be=safe\" role=\"link\" tabindex=\"0\">Be safe</a></i>? Or maybe <i><a class=\"squiffy-link link-section\" data-section=\"fun, be=fun\" role=\"link\" tabindex=\"0\">have fun</a>?</p>",
		'passages': {
		},
	},
	'safe': {
		'text': "<p>&quot;Be safe,&quot; she said at last.</p>\n<p>&quot;Don&#39;t worry,&quot; the turnip girl said. &quot;I can take care of myself.&quot;</p>\n<p>The very old witch wasn&#39;t so sure about that, but what could she say? She sat in her rocker and watched the turnip girl head out into the <a class=\"squiffy-link link-section\" data-section=\"world\" role=\"link\" tabindex=\"0\">world</a>.</p>",
		'passages': {
		},
	},
	'fun': {
		'text': "<p>&quot;Have fun,&quot; she said at last.</p>\n<p>&quot;I will try,&quot; the turnip girl said. &quot;I&#39;ll see you... when I see you?&quot;</p>\n<p>The very old witch nodded, and let the turnip girl head out into the <a class=\"squiffy-link link-section\" data-section=\"world\" role=\"link\" tabindex=\"0\">world</a>.</p>",
		'passages': {
		},
	},
	'world': {
		'clear': true,
		'text': "<p>The house was very empty without the turnip girl in it. It was too quiet and too dull and to her surprise the very old witch found herself more restless than before.</p>\n<p>Maybe the end of the world is coming, she reasoned, and spent some time drawing signs to post in the yard, signs that read, &quot;The end is nigh!&quot; and &quot;Reality is an illusion!&quot; and &quot;Keep off the grass!&quot;</p>\n<p>When that didn&#39;t help she took up knitting. But that only aggravated her carpal tunnel. She tried her hand at casting weather spells, conjuring up a magnificent snow storm in the middle of August, but that killed her garden and made her even more depressed. She spent some time screaming into her pillow and then throwing wads of newspaper at Hernando, which helped a little but not much.</p>\n<p><a class=\"squiffy-link link-section\" data-section=\"Four years\" role=\"link\" tabindex=\"0\">Four years</a> passed in this way.</p>",
		'passages': {
		},
	},
	'Four years': {
		'text': "<p>When four years had passed a car pulled up into the very old witch&#39;s lawn (there was no driveway). When the very old witch saw the car come rolling up onto her field of dandelions and clover she grew very angry. She came down off the porch to yell and shout and curse the very chassis of the car itself. As she did so one of the car&#39;s doors opened, and the turnip girl stepped out.</p>\n<p>&quot;Hello, Mama,&quot; she said.</p>\n<p>The very old witch lost her train of thought. &quot;Mama?&quot;</p>\n<p>&quot;I have to call you something. I came back to tell you I graduated.&quot;</p>\n<p>The very old witch was surprised. &quot;You actually went to college? And you didn&#39;t get made into turnip soup? How?&quot;</p>\n<p>&quot;Loans,&quot; the turnip girl said with a grimace. &quot;So many loans. But here, do you want to see my graduation?&quot; She held out a phone with a photo on its screen, but the very old witch could only make out some blurriness in the shape of possibly a turnip girl. Her eyes were not as good as they used to be. &quot;I graduated with a double major in economics and journalism. Do you want to see my clips?&quot;</p>\n<p>The very old witch glanced doubtfully at the phone, but the turnip girl brought out a big binder instead. She opened it up and inside were a number of newspaper clippings just big enough to be read.</p>\n<p>&quot;Neo Nazis March on Campus, by Turnip G. Witch,&quot; the very old witch read slowly. &quot;Allegations of Overseas Election Tampering. Police Tear-Gas Protestors. <a class=\"squiffy-link link-section\" data-section=\"What are these\" role=\"link\" tabindex=\"0\">What are these</a>?&quot;</p>",
		'passages': {
		},
	},
	'What are these': {
		'text': "<p>&quot;My clips,&quot; said the turnip girl. &quot;The world&#39;s a lot worse than you thought. There&#39;s white supremacists in the government and people are chasing away refugees and there have been seventeen trans women killed just so far this year. We&#39;re on our way right now to a counter-protest against the neo nazis.&quot;</p>\n<p>&quot;Sounds about right.&quot;</p>\n<p>&quot;I came back to ask if you want to do something about it.&quot;</p>\n<p>The very old witch laughed, but the turnip girl did not. &quot;If you are asking me to smite and blight nazis, my girl, you are bound to be disappointed. I gave up <a class=\"squiffy-link link-section\" data-section=\"meddling\" role=\"link\" tabindex=\"0\">meddling</a> in the affairs of humans.&quot;</p>",
		'passages': {
		},
	},
	'meddling': {
		'text': "<p>{if listen=yes:&quot;You told me once you gave that up because they were being so efficiently cruel to each other. Well, who is more effeciently cruel than a nazi?}{if listen=no:&quot;You wanted to let the humans do their own thing, and this is the thing they came up with. I can&#39;t let it get any worse.} {if explain=cant:It isn&#39;t right. The nazis don&#39;t get to be happy if they can&#39;t respect other people&#39;s humanity.}{if explain=wont:I may not be people but I know what&#39;s right. Just nazis being nazis is cruel to the people they hate. It&#39;s in their ideology!} I&#39;m not going to sit back and let those awful people make out like the people they hate are less than human. I can&#39;t let them do it.&quot;</p>\n<p>The very old witch smiled at her, feeling a little sadness in with the emptiness.</p>\n<p>&quot;I thought I told you to {if be=safe:to stay safe}{if be=fun:have fun}?&quot;</p>\n<p>The turnip girl took the very old witch&#39;s hand and <a class=\"squiffy-link link-section\" data-section=\"gave it a squeeze\" role=\"link\" tabindex=\"0\">gave it a squeeze</a>.</p>",
		'passages': {
		},
	},
	'gave it a squeeze': {
		'text': "<p>{if be=safe:&quot;I was,&quot; she said. &quot;And now I&#39;m going to keep the others safe too.&quot;}{if be=fun:&quot;I did,&quot; she said. &quot;And now I&#39;m here.&quot;}</p>\n<p>The very old witch looked down at the turnip girl&#39;s hand. &quot;I&#39;m not a strong witch anymore,&quot; she said. &quot;I can&#39;t see very well and I don&#39;t smite and blight like I used to.&quot;</p>\n<p>&quot;That&#39;s okay.&quot;</p>\n<p>&quot;I don&#39;t really know what I can do.&quot;</p>\n<p>&quot;That&#39;s okay too.&quot;</p>\n<p>&quot;And I&#39;m scared,&quot; the very old witch said very, very quietly.</p>\n<p>The turnip girl smiled at her.</p>\n<p>&quot;<a class=\"squiffy-link link-section\" data-section=\"Me too\" role=\"link\" tabindex=\"0\">Me too</a>,&quot; she said.</p>",
		'passages': {
		},
	},
	'Me too': {
		'text': "<p>The very old witch got into the car and was surprised to find it was already crowded with people, people in a wild variety of shades and abilities and hair styles.</p>\n<p>&quot;It&#39;s bad,&quot; one of the people in the front seat said as the turnip girl squeezed in beside her. &quot;I&#39;ve been keeping up with it on Twitter and it&#39;s so bad.&quot;</p>\n<p>&quot;That stands to reason, I guess,&quot; the turnip girl said. &quot;I mean, they are nazis. Let&#39;s get going.&quot;</p>\n<p>The car gave a lurch and rolled backward out of the lawn. The very old witch sent a mental order to Hernando to hold down the fort while she was gone and settled in for a long and uncomfortable ride.</p>\n<p>One thing she could say for it, though. She felt a <a class=\"squiffy-link link-section\" data-section=\"little less empty inside\" role=\"link\" tabindex=\"0\">little less empty inside</a>.</p>",
		'passages': {
		},
	},
	'little less empty inside': {
		'text': "<p>Maybe, she thought, watching as her turnip daughter and her friends organized to protect others, this was what had been missing all along.\n<img src=\"https://s2.postimg.org/94bfq5j7t/turnip2.jpg\" style=\"width:640px;height:388px;\">\n<sub></p>\n<p><a class=\"squiffy-link link-section\" data-section=\"about\" role=\"link\" tabindex=\"0\">about</a></sub></p>",
		'passages': {
		},
	},
	'about': {
		'text': "<p>text by <a href=\"http://whatfreshsilliness.weebly.com/\">Megan Stevens</a></p>\n<p>artwork by <a href=\"https://kelgraham.carbonmade.com/\">Kel Graham</a></p>",
		'passages': {
		},
	},
}
})();